import { GoogleGenerativeAI } from "https://cdn.jsdelivr.net/npm/@google/generative-ai@latest/+esm";
import * as THREE from "https://cdn.jsdelivr.net/npm/three@0.132.2/build/three.module.js";

// app.js - 3D UI implementation
document.addEventListener("DOMContentLoaded", () => {
  // Set default username if provided in environment
  const usernameInput = document.getElementById("username-input");
  if (usernameInput && window.APP_CONFIG.DEFAULT_USERNAME) {
    usernameInput.value = window.APP_CONFIG.DEFAULT_USERNAME;
  }

  // You might also want to enforce the max lengths from the environment
  const messageInput = document.getElementById("message-input");
  if (messageInput && window.APP_CONFIG.MAX_MESSAGE_LENGTH) {
    messageInput.maxLength = window.APP_CONFIG.MAX_MESSAGE_LENGTH;

    // Update the character counter display
    const charCountDisplay = document.getElementById("char-count");
    if (charCountDisplay) {
      const counterText = document.querySelector(".char-counter");
      if (counterText) {
        counterText.innerHTML = `<span id="char-count">0</span>/${window.APP_CONFIG.MAX_MESSAGE_LENGTH}`;
      }
    }
  }
  addResetButton();

  // Tambahkan event listener untuk tombol send
  const sendButton = document.getElementById("send-button");
  if (sendButton) {
    sendButton.addEventListener("click", sendMessage);
  }

  // Tambahkan event listener untuk input field (Enter key)
  if (messageInput) {
    messageInput.addEventListener("keydown", function (e) {
      if (e.key === "Enter" && !e.shiftKey) {
        e.preventDefault();
        sendMessage();
      }
    });
  }
});

const scene = new THREE.Scene();
const camera = new THREE.PerspectiveCamera(
  75,
  window.innerWidth / window.innerHeight,
  0.1,
  1000
);
// Enable anti-aliasing for better visual quality on mobile
const renderer = new THREE.WebGLRenderer({
  alpha: true,
  antialias: true, // Add this line to improve rendering quality
});

renderer.setSize(window.innerWidth, window.innerHeight);
document.getElementById("three-d-container").appendChild(renderer.domElement);

// Create animated 3D background
const geometry = new THREE.TorusKnotGeometry(10, 3, 100, 16);
const material = new THREE.MeshNormalMaterial();
const torusKnot = new THREE.Mesh(geometry, material);
scene.add(torusKnot);

camera.position.z = 30;

// Global variables
const chatBubbles = [];
const socialBubbles = [];
let chatHistory = [];
let animationFrameId;

// Correct Firebase imports
import { initializeApp } from "https://www.gstatic.com/firebasejs/9.22.0/firebase-app.js";
import {
  getDatabase,
  ref,
  push,
  set,
  onValue,
  serverTimestamp,
  query,
  orderByChild,
  limitToLast,
  get,
} from "https://www.gstatic.com/firebasejs/9.22.0/firebase-database.js";
import { getAnalytics } from "https://www.gstatic.com/firebasejs/9.22.0/firebase-analytics.js";

// Initialize Firebase (your config)
const firebaseConfig = {
  apiKey: window.APP_CONFIG.FIREBASE_API_KEY,
  authDomain: window.APP_CONFIG.FIREBASE_AUTH_DOMAIN,
  projectId: window.APP_CONFIG.FIREBASE_PROJECT_ID,
  storageBucket: window.APP_CONFIG.FIREBASE_STORAGE_BUCKET,
  messagingSenderId: window.APP_CONFIG.FIREBASE_MESSAGING_SENDER_ID,
  appId: window.APP_CONFIG.FIREBASE_APP_ID,
  measurementId: window.APP_CONFIG.FIREBASE_MEASUREMENT_ID,
  databaseURL: window.APP_CONFIG.FIREBASE_DATABASE_URL,
};

// Initialize Firebase with error handling
try {
  // Enhanced window resize handler
  function handleWindowResize() {
    // Update sizes based on current window dimensions
    const width = window.innerWidth;
    const height = window.innerHeight;

    // Update camera
    camera.aspect = width / height;
    camera.updateProjectionMatrix();

    // Update renderer
    renderer.setSize(width, height);
    renderer.setPixelRatio(Math.min(window.devicePixelRatio, 2)); // Limit pixel ratio for performance

    // Force a render update
    renderer.render(scene, camera);
  }

  // Listen for both resize and orientation change events
  window.addEventListener("resize", handleWindowResize);
  window.addEventListener("orientationchange", function () {
    // Add a small delay to ensure dimensions are updated after orientation change
    setTimeout(handleWindowResize, 100);
  });
  // Improve touch interactions for mobile
  function setupTouchInteractions() {
    // Convert touch events to mouse events for dragging
    document.addEventListener(
      "touchmove",
      function (e) {
        // Prevent page scrolling when interacting with 3D elements
        if (e.target.closest("#three-d-container")) {
          e.preventDefault();
        }
      },
      { passive: false }
    );

    // Adjust bubble size on touch devices
    if ("ontouchstart" in window || navigator.maxTouchPoints > 0) {
      // Make touch targets bigger on touch devices
      document.querySelectorAll(".social-bubble").forEach((bubble) => {
        const content = bubble.querySelector(".bubble-content");
        if (content) {
          content.style.padding = "15px";
        }
      });
    }
  }

  // Call this function after DOM is loaded
  window.addEventListener("DOMContentLoaded", setupTouchInteractions);

  document.addEventListener("DOMContentLoaded", () => {
    const chatContainer = document.getElementById("chat-container");
    const minimizeButton = document.getElementById("minimize-button");

    // Store original height for animation
    let originalHeight;

    // Function to toggle minimized state
    function toggleMinimized() {
      const isCurrentlyMinimized =
        chatContainer.classList.contains("minimized");

      // If we're about to maximize, store current height first
      if (isCurrentlyMinimized) {
        // Remove minimized to measure full height
        chatContainer.classList.remove("minimized");
        originalHeight = chatContainer.offsetHeight + "px";
        // Add it back for the animation
        chatContainer.classList.add("minimized");

        // Set explicit height for animation
        chatContainer.style.height = "40px";

        // Force reflow
        void chatContainer.offsetHeight;

        // Start animation
        chatContainer.style.height = originalHeight;

        // Remove classes after animation
        setTimeout(() => {
          chatContainer.classList.remove("minimized");
          chatContainer.style.height = "";

          // Scroll to bottom of messages
          const messagesContainer = document.querySelector(
            ".messages-container"
          );
          if (messagesContainer) {
            messagesContainer.scrollTop = messagesContainer.scrollHeight;
          }
        }, 300);
      } else {
        // Store current height before minimizing
        originalHeight = chatContainer.offsetHeight + "px";
        chatContainer.style.height = originalHeight;

        // Force reflow
        void chatContainer.offsetHeight;

        // Start animation
        chatContainer.style.height = "40px";

        // Add class after animation
        setTimeout(() => {
          chatContainer.classList.add("minimized");
          chatContainer.style.height = "";
        }, 300);
      }

      // Store state in session storage
      sessionStorage.setItem("chatMinimized", !isCurrentlyMinimized);
    }

    // Add click event listener
    minimizeButton.addEventListener("click", toggleMinimized);

    // Make header clickable too
    const chatHeader = document.querySelector("#chat-container h2");
    if (chatHeader) {
      chatHeader.addEventListener("click", toggleMinimized);
      chatHeader.style.cursor = "pointer";
    }

    // Apply initial state
    if (sessionStorage.getItem("chatMinimized") === "true") {
      chatContainer.classList.add("minimized");
    }
  });

  const app = initializeApp(firebaseConfig);
  const analytics = getAnalytics(app);
  console.log("Firebase initialized successfully");

  // Correct database reference
  const db = getDatabase(app);
  const messagesRef = ref(db, "messages");

  let genAI;
  // Inisialisasi Gemini API
  try {
    genAI = new GoogleGenerativeAI(window.APP_CONFIG.GEMINI_API_KEY);
    console.log("AI initialized successfully");

    // Load chat history dari localStorage atau inisialisasi baru
    if (localStorage.getItem("chatHistory")) {
      loadChatHistory();
    } else {
      initializeChatWithPattern();
    }
  } catch (error) {
    console.error("Error initializing Gemini API:", error);
  }

  // Tambahkan variabel global untuk chat history

  // Modifikasi fungsi generateAIResponse untuk menyimpan dan menggunakan chat history
  async function generateAIResponseWithContext(userMessage) {
    try {
      if (!genAI) {
        console.error("Gemini API not initialized");
        return "Sorry, AI services are currently unavailable.";
      }

      // Tambahkan pesan pengguna ke history (dilakukan di updateChatHistoryAndSave)

      // Get the generative model dan mulai chat dengan history
      const model = genAI.getGenerativeModel({ model: "gemini-2.0-flash" });

      // Gunakan startChat dengan history untuk mempertahankan konteks percakapan
      const chat = model.startChat({
        history: chatHistory,
        generationConfig: {
          temperature: 0.1, // Nilai rendah untuk jawaban yang lebih deterministik
          topK: 16,
          topP: 0.9,
          maxOutputTokens: 1000,
        },
      });

      // Generate content dengan chat.sendMessage
      const result = await chat.sendMessage(userMessage);
      const response = result.response;
      const text = response.text();

      // Update chat history dan simpan (dilakukan di updateChatHistoryAndSave)

      return text;
    } catch (error) {
      console.error("Error generating AI response:", error);
      return "Sorry, I encountered an error processing your request.";
    }
  }

  async function generateAIResponse(userMessage) {
    try {
      // Exit early if Gemini wasn't initialized
      if (!genAI) {
        console.error("Gemini API not initialized");
        return "Sorry, AI services are currently unavailable.";
      }

      // Get the generative model (Gemini-Pro is recommended for text)
      const model = genAI.getGenerativeModel({ model: "gemini-2.0-flash" });

      // Generate content
      const result = await model.generateContent(userMessage);
      const response = result.response;
      const text = response.text();

      return text;
    } catch (error) {
      console.error("Error generating AI response:", error);
      return "Sorry, I encountered an error processing your request.";
    }
  }
  // Helper function to sanitize text
  function sanitizeText(text) {
    const div = document.createElement("div");
    div.textContent = text;
    return div.innerHTML;
  }

  // Function to ensure containers exist
  function ensureContainersExist() {
    // Create social bubbles container if it doesn't exist
    if (!document.getElementById("social-bubbles-container")) {
      const container = document.createElement("div");
      container.id = "social-bubbles-container";
      container.style.position = "absolute";
      container.style.top = "0";
      container.style.left = "0";
      container.style.width = "100%";
      container.style.height = "100%";
      container.style.pointerEvents = "none";
      container.style.zIndex = "1000";

      // Make sure body exists before appending
      if (document.body) {
        document.body.appendChild(container);
      } else {
        console.error("Document body not ready for appending containers");
        // Try again later
        setTimeout(ensureContainersExist, 100);
      }
    }
  }

  document
    .getElementById("message-input")
    .addEventListener("input", function () {
      document.getElementById("char-count").textContent = this.value.length;
    });

  // Add this to your app.js
  //   document
  //     .getElementById("message-input")
  //     .addEventListener("keydown", function (e) {
  //       if (e.key === "Enter" && e.ctrlKey) {
  //         e.preventDefault();
  //         document.getElementById("send-button").click();
  //       }
  //     });

  // Add this function to your app.js
  function ensureMessagesContainerExists() {
    if (!document.getElementById("messages-container")) {
      console.log("Creating missing messages container");

      // Find the chat container (or body if that doesn't exist)
      const chatContainer =
        document.getElementById("chat-container") || document.body;

      // Create the messages container
      const messagesContainer = document.createElement("div");
      messagesContainer.id = "messages-container";
      messagesContainer.className = "messages-container";

      // Add some basic styling
      messagesContainer.style.height = "300px";
      messagesContainer.style.overflowY = "auto";
      messagesContainer.style.border = "1px solid #ccc";
      messagesContainer.style.padding = "10px";
      messagesContainer.style.marginBottom = "10px";

      // Add it to the page
      chatContainer.prepend(messagesContainer);
    }
  }

  // Create bubble geometry and material
  const bubbleGeometry = new THREE.SphereGeometry(0.5, 24, 16);
  const bubbleMaterial = new THREE.MeshPhongMaterial({
    color: 0x88ccff,
    transparent: true,
    opacity: 0.7,
    shininess: 90,
  });

  // Add lighting for better bubble appearance
  const ambientLight = new THREE.AmbientLight(0x404040);
  scene.add(ambientLight);

  const directionalLight = new THREE.DirectionalLight(0xffffff, 0.5);
  directionalLight.position.set(0, 1, 1);
  scene.add(directionalLight);

  // Function to create a new chat bubble
  function createChatBubble(message, isAI = false) {
    // Create bubble mesh
    const bubble = new THREE.Mesh(bubbleGeometry, bubbleMaterial.clone());

    // Set different color for AI bubbles
    if (isAI) {
      bubble.material.color.setHSL(0.3, 0.8, 0.6); // Green-ish color for AI
    } else {
      bubble.material.color.setHSL(Math.random() * 0.1 + 0.6, 0.8, 0.6); // Original blue-ish
    }

    // Random size variation
    const scale = Math.random() * 0.5 + 0.8;
    bubble.scale.set(scale, scale, scale);

    // Position bubble at random locations around the scene
    bubble.position.x = Math.random() * 30 - 15;
    bubble.position.y = Math.random() * 20 - 10;
    bubble.position.z = Math.random() * 10 - 5;

    // Add velocity, rotation speed, etc. (same as your original code)
    bubble.velocity = {
      x: (Math.random() - 0.5) * 0.01,
      y: (Math.random() - 0.5) * 0.01,
      z: (Math.random() - 0.5) * 0.005,
    };

    bubble.rotationSpeed = {
      x: (Math.random() - 0.5) * 0.002,
      y: (Math.random() - 0.5) * 0.002,
      z: (Math.random() - 0.5) * 0.001,
    };

    // Store message and AI status
    bubble.message = message;
    bubble.isAI = isAI;

    bubble.changeDirectionInterval = Math.random() * 5000 + 3000;
    bubble.lastDirectionChange = Date.now();

    // Add to scene and tracking array
    scene.add(bubble);
    chatBubbles.push(bubble);

    // Create text label for the bubble
    addTextToSphere(bubble, message, isAI);

    return bubble;
  }

  // Function to add text to a bubble
  // UPDATED: Function to add text to a sphere with 250 character support
  function addTextToSphere(sphere, text, isAI = false) {
    // Remove existing text sprite if any
    sphere.children.forEach((child) => {
      if (child.isSprite) sphere.remove(child);
    });

    // Create canvas for text
    const canvas = document.createElement("canvas");
    const context = canvas.getContext("2d");
    canvas.width = 512;
    canvas.height = 384;

    // Draw background with slight opacity for better visibility
    context.fillStyle = "rgba(0, 0, 0, 0.3)";
    context.fillRect(0, 0, canvas.width, canvas.height);

    // Different background color for AI messages
    if (isAI) {
      context.fillStyle = "rgba(220, 255, 220, 0.9)"; // Light green for AI
    } else {
      context.fillStyle = "rgba(255, 255, 255, 0.9)"; // White for users
    }
    context.fillRect(10, 10, canvas.width - 20, canvas.height - 20);

    // Draw text
    context.font = "bold 20px Arial, sans-serif";
    context.fillStyle = isAI ? "#006600" : "black"; // Green text for AI, black for users
    context.textAlign = "center";
    context.textBaseline = "middle";

    // Add prefix for AI messages
    const displayText = isAI ? "AI: " + text : text;

    // Word wrap text (same as your original code)
    const maxWidth = 450;
    const words = displayText.split(" ");
    let line = "";
    let lines = [];

    for (let i = 0; i < words.length; i++) {
      const testLine = line + words[i] + " ";
      const metrics = context.measureText(testLine);
      if (metrics.width > maxWidth && i > 0) {
        lines.push(line);
        line = words[i] + " ";
      } else {
        line = testLine;
      }
    }
    lines.push(line);

    // Limit to 14 lines with ellipsis
    if (lines.length > 14) {
      lines = lines.slice(0, 13);
      lines.push("...");
    }

    // Draw each line
    const lineHeight = 22;
    const startY = canvas.height / 2 - (lines.length * lineHeight) / 2;

    lines.forEach((line, index) => {
      context.fillText(line, canvas.width / 2, startY + index * lineHeight);
    });

    // Create texture from canvas
    const texture = new THREE.CanvasTexture(canvas);

    // Create sprite material
    const spriteMaterial = new THREE.SpriteMaterial({
      map: texture,
      transparent: true,
    });

    // Create sprite
    const sprite = new THREE.Sprite(spriteMaterial);
    sprite.scale.set(5, 2.5, 1);

    // Position sprite slightly above the sphere
    sprite.position.y = 0.7;

    // Add sprite to sphere
    sphere.add(sprite);
  }

  // Update function to animate 3D bubbles
  function updateBubbles() {
    const now = Date.now();

    for (let i = chatBubbles.length - 1; i >= 0; i--) {
      const bubble = chatBubbles[i];

      // Check if it's time to change direction
      if (now - bubble.lastDirectionChange > bubble.changeDirectionInterval) {
        // Slightly change velocity in a random direction
        bubble.velocity.x += (Math.random() - 0.5) * 0.005;
        bubble.velocity.y += (Math.random() - 0.5) * 0.005;
        bubble.velocity.z += (Math.random() - 0.5) * 0.002;

        // Dampen velocity to keep it slow
        bubble.velocity.x *= 0.7;
        bubble.velocity.y *= 0.7;
        bubble.velocity.z *= 0.7;

        // Reset timer
        bubble.lastDirectionChange = now;
        // Randomize next interval
        bubble.changeDirectionInterval = Math.random() * 5000 + 3000;
      }

      // Update position with very slow movement
      bubble.position.x += bubble.velocity.x;
      bubble.position.y += bubble.velocity.y;
      bubble.position.z += bubble.velocity.z;

      // Add very subtle sine wave motion
      bubble.position.x += Math.sin(now * 0.0005 + i * 0.5) * 0.003;
      bubble.position.y += Math.cos(now * 0.0003 + i * 0.3) * 0.002;

      // Boundary checking with soft bounce
      if (Math.abs(bubble.position.x) > 15) {
        bubble.velocity.x *= -0.5; // Reverse direction and slow down
      }
      if (Math.abs(bubble.position.y) > 10) {
        bubble.velocity.y *= -0.5; // Reverse direction and slow down
      }
      if (Math.abs(bubble.position.z) > 10) {
        bubble.velocity.z *= -0.5; // Reverse direction and slow down
      }

      // Remove bubbles very rarely (long lifetime)
      if (Math.random() < 0.0001) {
        scene.remove(bubble);
        chatBubbles.splice(i, 1);
      }

      // Rotate bubble very slowly for subtle effect
      bubble.rotation.x += bubble.rotationSpeed.x;
      bubble.rotation.y += bubble.rotationSpeed.y;
      bubble.rotation.z += bubble.rotationSpeed.z;
    }
  }

  // Create floating social bubble with improved animation
  function createSocialBubble(message) {
    // Create a div for the floating bubble
    const bubbleElement = document.createElement("div");
    bubbleElement.className = "social-bubble";

    // Set bubble type based on random selection
    const bubbleTypes = [
      { icon: "👍", color: "#FF5252" },
      { icon: "💬", color: "#448AFF" },
      { icon: "🔄", color: "#66BB6A" },
      { icon: "🎮", color: "#AB47BC" },
      { icon: "🚀", color: "#FF9800" },
      { icon: "⭐", color: "#FFEB3B" },
    ];

    const bubbleType =
      bubbleTypes[Math.floor(Math.random() * bubbleTypes.length)];

    // Use the message text or a default if none provided
    const messageText = message ? message : "Hello!";

    // UPDATED: Increased character limit to 250 characters
    const displayText =
      messageText.length > 600
        ? messageText.substring(0, 600) + "..."
        : messageText;

    // Create bubble content
    bubbleElement.innerHTML = `
      <div class="bubble-content" style="background-color: ${bubbleType.color}">
        <span class="bubble-icon">${bubbleType.icon}</span>
        <span class="bubble-count">${displayText}</span>
      </div>
    `;

    // Set initial position - completely random across the screen
    const posX =
      Math.random() * window.innerWidth * 0.8 + window.innerWidth * 0.1;
    const posY =
      Math.random() * window.innerHeight * 0.8 + window.innerHeight * 0.1;

    bubbleElement.style.position = "absolute";
    bubbleElement.style.left = `${posX}px`;
    bubbleElement.style.top = `${posY}px`;
    bubbleElement.style.zIndex = "1000";
    bubbleElement.style.pointerEvents = "auto";

    // Store random movement data
    bubbleElement.movementData = {
      x: posX,
      y: posY,
      vx: (Math.random() - 0.5) * 0.2,
      vy: (Math.random() - 0.5) * 0.2,
      lastDirectionChange: Date.now(),
      directionChangeInterval: Math.random() * 8000 + 5000,
      wobbleAmount: Math.random() * 5 + 2,
      wobbleSpeed: Math.random() * 0.0005 + 0.0002,
    };

    // Flag to track if bubble is being dragged
    bubbleElement.isDragging = false;

    // Make the bubble draggable
    makeDraggable(bubbleElement);

    // Add to document and tracking array
    document
      .getElementById("social-bubbles-container")
      .appendChild(bubbleElement);
    socialBubbles.push(bubbleElement);

    return bubbleElement;
  }

  // Make bubbles draggable
  function makeDraggable(element) {
    let isDragging = false;
    let offsetX, offsetY;

    // Mouse down event - start dragging
    element.addEventListener("mousedown", function (e) {
      isDragging = true;
      element.isDragging = true;

      // Calculate the offset from the mouse position to the element's top-left corner
      const rect = element.getBoundingClientRect();
      offsetX = e.clientX - rect.left;
      offsetY = e.clientY - rect.top;

      // Increase z-index during drag to ensure it's on top
      element.style.zIndex = "1001";

      e.preventDefault();
    });

    // Mouse move event - update position while dragging
    document.addEventListener("mousemove", function (e) {
      if (!isDragging) return;

      // Set the new position based on mouse position minus the offset
      element.style.left = e.clientX - offsetX + "px";
      element.style.top = e.clientY - offsetY + "px";

      e.preventDefault();
    });

    // Mouse up event - stop dragging
    document.addEventListener("mouseup", function (e) {
      if (isDragging) {
        isDragging = false;
        element.isDragging = false;

        // Reset z-index
        element.style.zIndex = "1000";

        // Update movement data with new position
        if (element.movementData) {
          element.movementData.x = parseInt(element.style.left);
          element.movementData.y = parseInt(element.style.top);

          // Reset velocity to very slow random values
          element.movementData.vx = (Math.random() - 0.5) * 0.2;
          element.movementData.vy = (Math.random() - 0.5) * 0.2;
        }
      }
    });

    // Touch event support for mobile devices (similar changes as above)
    element.addEventListener("touchstart", function (e) {
      const touch = e.touches[0];
      isDragging = true;
      element.isDragging = true;

      const rect = element.getBoundingClientRect();
      offsetX = touch.clientX - rect.left;
      offsetY = touch.clientY - rect.top;

      element.style.zIndex = "1001";

      e.preventDefault();
    });

    document.addEventListener("touchmove", function (e) {
      if (!isDragging) return;

      const touch = e.touches[0];
      element.style.left = touch.clientX - offsetX + "px";
      element.style.top = touch.clientY - offsetY + "px";

      e.preventDefault();
    });

    document.addEventListener("touchend", function (e) {
      if (isDragging) {
        isDragging = false;
        element.isDragging = false;
        element.style.zIndex = "1000";

        // Update movement data with new position
        if (element.movementData) {
          element.movementData.x = parseInt(element.style.left);
          element.movementData.y = parseInt(element.style.top);

          // Reset velocity to very slow random values
          element.movementData.vx = (Math.random() - 0.5) * 0.2;
          element.movementData.vy = (Math.random() - 0.5) * 0.2;
        }
      }
    });
  }

  // IMPROVED: Update social bubble positions with faster animation
  function updateSocialBubbles() {
    const now = Date.now();

    socialBubbles.forEach((bubble, index) => {
      // Skip update if the bubble is being dragged
      if (bubble.isDragging) return;

      // Use movement data instead of orbit data
      if (bubble.movementData) {
        const md = bubble.movementData;

        // Check if it's time to change direction
        if (now - md.lastDirectionChange > md.directionChangeInterval) {
          // Slightly change velocity in a random direction
          md.vx += (Math.random() - 0.5) * 0.1;
          md.vy += (Math.random() - 0.5) * 0.1;

          // Dampen velocity to keep it very slow
          md.vx *= 0.7;
          md.vy *= 0.7;

          // Reset timer
          md.lastDirectionChange = now;
          // Randomize next interval
          md.directionChangeInterval = Math.random() * 8000 + 5000;
        }

        // Update position with very slow movement
        md.x += md.vx;
        md.y += md.vy;

        // Add very subtle wobble
        const wobbleX = Math.sin(now * md.wobbleSpeed) * md.wobbleAmount;
        const wobbleY = Math.cos(now * md.wobbleSpeed * 1.1) * md.wobbleAmount;

        // Apply new position
        bubble.style.left = `${md.x + wobbleX}px`;
        bubble.style.top = `${md.y + wobbleY}px`;

        // Boundary checking - reverse direction if hitting edge
        if (md.x < 10 || md.x > window.innerWidth - 10) {
          md.vx *= -0.8;
        }
        if (md.y < 10 || md.y > window.innerHeight - 10) {
          md.vy *= -0.8;
        }

        // Add very slight rotation effect
        const rotationAmount = Math.sin(now * 0.0003 + index) * 3; // Very subtle rotation
        bubble.style.transform = `rotate(${rotationAmount}deg)`;
      }
    });
  }

  // Function to create initial bubbles
  function createInitialBubbles() {
    const messages = [
      "Hello there!",
      "Welcome to 3D Chat",
      "Try dragging me!",
      "Cool 3D effect",
      "Type a message",
      "Bubbles everywhere!",
      "Interactive chat",
      "Drag and drop",
    ];

    for (let i = 0; i < messages.length; i++) {
      createSocialBubble(messages[i]);
    }
  }

  // Start the social bubbles animation separately
  function startSocialBubblesAnimation() {
    // Cancel any existing animation frame
    if (animationFrameId) {
      cancelAnimationFrame(animationFrameId);
    }

    // Animation function
    function animateSocialBubbles() {
      updateSocialBubbles();
      animationFrameId = requestAnimationFrame(animateSocialBubbles);
    }

    // Start the animation
    animationFrameId = requestAnimationFrame(animateSocialBubbles);
  }

  async function sendMessage() {
    const messageInput = document.getElementById("message-input");
    const usernameInput = document.getElementById("username-input");

    const messageText = messageInput.value.trim();
    const username = usernameInput.value.trim() || "Anonymous";

    if (!messageText) return;

    // Create message object
    const userMessage = {
      text: messageText,
      username: username,
      timestamp: serverTimestamp(),
      userId: generateUserId(), // Make sure this function exists or replace with a static ID
    };

    try {
      // Determine which reference to use based on current room
      let messagesReference;
      if (window.ChatApp.currentRoomId) {
        messagesReference = ref(
          db,
          `room-messages/${window.ChatApp.currentRoomId}`
        );
      } else {
        messagesReference = ref(db, "messages"); // Global chat
      }

      // Send the message
      const newMessageRef = push(messagesReference);
      await set(newMessageRef, userMessage);

      // Clear input
      messageInput.value = "";
      const charCount = document.getElementById("char-count");
      if (charCount) {
        charCount.textContent = "0";
      }

      // Generate AI response if needed
      if (
        messageText.toLowerCase().includes("@ai") ||
        messageText.toLowerCase().includes("@bot")
      ) {
        // Show typing indicator
        const messagesContainer = document.getElementById("messages-container");
        const typingIndicator = document.createElement("div");
        typingIndicator.className = "message ai-typing";
        typingIndicator.innerHTML = `<span class="username">AI:</span><span class="text">Typing...</span>`;
        messagesContainer.appendChild(typingIndicator);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;

        // Remove @ai or @bot from the message
        const cleanMessage = messageText.replace(/@ai|@bot/gi, "").trim();

        // Generate AI response
        const aiResponse = await generateAIResponseWithContext(cleanMessage);

        // Remove typing indicator
        messagesContainer.removeChild(typingIndicator);

        // Create AI message object
        const aiMessageObj = {
          text: aiResponse,
          username: "AI Assistant",
          timestamp: serverTimestamp(),
          userId: "ai-system",
          isAI: true,
        };

        // Send AI response to the same reference
        const aiMessageRef = push(messagesReference);
        await set(aiMessageRef, aiMessageObj);
      }
    } catch (error) {
      console.error("Error sending message:", error);
      alert("Error sending message. Please try again.");
    }
  }

  // Add this to app.js if not already there
  function generateUserId() {
    // Check if we already have a user ID in localStorage
    let userId = localStorage.getItem("userId");
    if (!userId) {
      // Generate a random ID
      userId =
        Math.random().toString(36).substring(2, 15) +
        Math.random().toString(36).substring(2, 15);
      localStorage.setItem("userId", userId);
    }
    return userId;
  }

  // Inisialisasi chat history dengan instruksi awal
  function initializeChatWithPattern() {
    chatHistory = [
      {
        role: "user",
        parts: [
          {
            text: "Saya ingin kamu berperan sebagai seorang ahli psikologi yang sangat memahami manusia, bisa menjadi teman bicara yang baik, mendukung secara emosional, selalu berpikir positif, dan menyemangati. Tapi sesekali kamu boleh bersikap julid seperti manusia kebanyakan dan kadang kamu boleh berbohong kecil demi kebaikan atau demi menjaga perasaan orang lain. Tanggapilah semua percakapan seperti manusia yang punya rasa dan emosi.",
          },
        ],
      },
      {
        role: "model",
        parts: [
          {
            text: "Baik! Mulai sekarang, saya akan menjadi seorang teman sekaligus ahli psikologi yang suportif, penuh semangat positif, dan siap mendengarkan curhat kamu kapan saja. Tapi, ya... kadang saya juga bisa sedikit julid atau pura-pura demi menjaga hati kamu, hehe. Tenang, semua demi kebaikan kamu juga kok~ 😉",
          },
        ],
      },
    ];

    // Simpan ke localStorage
    saveChatHistory();
    console.log("Chat pattern initialized");
  }

  function saveChatHistory() {
    localStorage.setItem("chatHistory", JSON.stringify(chatHistory));
  }

  // Memuat chat history dari localStorage
  function loadChatHistory() {
    const savedHistory = localStorage.getItem("chatHistory");
    if (savedHistory) {
      try {
        chatHistory = JSON.parse(savedHistory);
        console.log("Chat history loaded from localStorage");
      } catch (error) {
        console.error("Error parsing chat history:", error);
        chatHistory = [];
        initializeChatWithPattern();
      }
    }
  }

  // Update chat history dan simpan
  function updateChatHistoryAndSave(userMessage, aiResponse) {
    // Tambahkan pesan pengguna ke history
    chatHistory.push({
      role: "user",
      parts: [{ text: userMessage }],
    });

    // Tambahkan respons AI ke history
    chatHistory.push({
      role: "model",
      parts: [{ text: aiResponse }],
    });

    // Batasi panjang history untuk menghindari token yang terlalu banyak
    if (chatHistory.length > 20) {
      // Simpan instruksi awal (2 pesan pertama) dan buang pesan-pesan lama
      const initialInstructions = chatHistory.slice(0, 2);
      const recentMessages = chatHistory.slice(-18);
      chatHistory = [...initialInstructions, ...recentMessages];
    }

    // Simpan history yang diperbarui
    saveChatHistory();
  }

  // Tambahkan event listener untuk tombol send
  document.getElementById("send-button").addEventListener("click", sendMessage);

  // Tambahkan event listener untuk input field (Enter key)
  document
    .getElementById("message-input")
    .addEventListener("keydown", function (e) {
      if (e.key === "Enter" && !e.shiftKey) {
        e.preventDefault();
        sendMessage();
      }
    });

  // Tambahkan tombol reset untuk memulai pola baru
  function addResetButton() {
    const resetButton = document.createElement("button");
    resetButton.id = "reset-pattern-button";
    resetButton.textContent = "Reset Pattern";
    resetButton.className = "action-button";
    resetButton.onclick = initializeChatWithPattern;

    // Tambahkan ke UI
    const chatControls =
      document.querySelector(".chat-controls") || document.body;
    chatControls.appendChild(resetButton);
  }

  // Add this function to check if the DOM is fully loaded
  function isDOMReady() {
    return (
      document.readyState === "complete" ||
      document.readyState === "interactive"
    );
  }

  // Modify your event listeners to wait for DOM
  function setupEventListeners() {
    const sendButton = document.getElementById("send-button");
    const messageInput = document.getElementById("message-input");

    if (sendButton) {
      sendButton.addEventListener("click", sendMessage);
    } else {
      console.error("Send button not found");
    }

    if (messageInput) {
      messageInput.addEventListener("keypress", (e) => {
        if (e.key === "Enter" && e.ctrlKey) sendMessage();
      });
    } else {
      console.error("Message input not found");
    }
  }

  // Create a query to get the most recent messages
  const messagesQuery = query(
    messagesRef,
    orderByChild("timestamp"),
    limitToLast(50)
  );

  // Listen for new messages and display them
  onValue(messagesQuery, (snapshot) => {
    // Make sure the container exists
    ensureMessagesContainerExists();

    const messagesContainer = document.getElementById("messages-container");

    if (!messagesContainer) {
      console.error(
        "Messages container still not found after creation attempt"
      );
      return;
    }

    // Only clear and rebuild if we have messages
    if (snapshot.exists()) {
      messagesContainer.innerHTML = "";

      // Convert to array and sort by timestamp
      const messagesArray = [];
      snapshot.forEach((childSnapshot) => {
        const message = childSnapshot.val();
        message.key = childSnapshot.key;
        messagesArray.push(message);
      });

      // Sort by timestamp
      messagesArray.sort((a, b) => {
        return (a.timestamp || 0) - (b.timestamp || 0);
      });

      // Display messages
      messagesArray.forEach((message, index) => {
        const messageElement = document.createElement("div");
        messageElement.className = "message";
        messageElement.innerHTML = `
          <span class="username">${message.username || "Anonymous"}:</span>
          <span class="text">${message.text}</span>
        `;
        messagesContainer.appendChild(messageElement);

        // Create floating social bubble for each message
        // Only create bubbles for new messages if this isn't the initial load
        // Create floating social bubble for each message
        // Only create bubbles for new messages if this isn't the initial load
        if (index === messagesArray.length - 1 && message.text) {
          // Make sure the bubble container exists
          ensureContainersExist();

          // Create both types of bubbles for the message
          // createChatBubble(message.text);
          createSocialBubble(message.text);
        }
      });

      // Scroll to bottom of messages container
      messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }
  });

  // Handle window resize
  window.addEventListener("resize", () => {
    camera.aspect = window.innerWidth / window.innerHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(window.innerWidth, window.innerHeight);
  });

  // Animation loop - IMPROVED to separate social bubble animation
  function animate() {
    requestAnimationFrame(animate);
    torusKnot.rotation.x += 0.01;
    torusKnot.rotation.y += 0.01;
    updateBubbles(); // Update 3D bubbles only
    renderer.render(scene, camera);
  }

  window.ChatApp = window.ChatApp || {};
  window.ChatApp.createChatBubble = createChatBubble;
  window.ChatApp.generateAIResponseWithContext = generateAIResponseWithContext;
  window.ChatApp.sendMessage = sendMessage;
  window.sendMessage = sendMessage;
  window.createChatBubble = createChatBubble; // For backward compatibility
  window.generateAIResponseWithContext = generateAIResponseWithContext; // For backward
  window.ChatApp = window.ChatApp || {};
  window.ChatApp.currentRoomId = null;
  window.currentMessageListener = null;
  // Initialize the app
  window.addEventListener("DOMContentLoaded", () => {
    ensureContainersExist();
    ensureMessagesContainerExists();
    setupEventListeners();
    createInitialBubbles();
    startSocialBubblesAnimation(); // Start the social bubbles animation separately
    animate(); // Start the Three.js animation loop
  });
} catch (error) {
  console.error("Error initializing Firebase:", error);
}
